/** @file   modifyevent.cpp
 * @brief   Implementation of ModifyEvent class
 * @version $Revision: 1.2 $
 * @date    $Date: 2006/07/27 20:31:51 $
 * @author  Tomi Lamminsaari
 */


#include "modifyevent.h"
#include "www_map.h"
#include "soundsamples.h"
#include "settings.h"
#include "www_assert.h"
using std::istream;
using std::string;
using namespace eng2d;

namespace WeWantWar {


/** Constructor.
 */
ModifyEvent::ModifyEvent() :
  DynamicEvent()
{
}



/** Destructor.
 */
ModifyEvent::~ModifyEvent()
{
  for (int i=0; i < m_modifications.size(); i++) {
    delete m_modifications.at(i);
  }
  m_modifications.clear();
}



/** Reads the data of this event.
 */
int ModifyEvent::readData( istream& fin )
{
  LOG_MESSAGE( "Reading Modify-event" );
  string tmp;
  
  while ( true ) {
    if ( fin.eof() ) {
      break;
    }
    
    fin >> tmp;
    
    if ( tmp == "</modify>" ) {
      return 0;
      
    } else if ( tmp == "at:" ) {
      this->createModification( fin );
      
    } else if ( tmp == "hline:" ) {
      this->createHLineModification( fin );
      
    } else if ( tmp == "vline:" ) {
      this->createVLineModification( fin );
      
    } else if ( tmp == "sound:" ) {
      fin >> DynamicEvent::m_sound;
      
    } else if ( tmp == "delay:" ) {
      fin >> DynamicEvent::m_counter;
      
    } else if ( tmp == "difficulty:" ) {
      fin >> tmp;
      DynamicEvent::m_difficulty = 0;
      if ( tmp.find("e",0) != string::npos ) {
        DynamicEvent::m_difficulty |= KBitmaskEasy;
      }
      if ( tmp.find("n",0) != string::npos ) {
        DynamicEvent::m_difficulty |= KBitmaskNormal;
      }
      if ( tmp.find("h",0) != string::npos ) {
        DynamicEvent::m_difficulty |= KBitmaskHard;
      }
      
    } else if ( tmp == "<copyblock>" ) {
      if ( this->readCopyBlock( fin ) != 0 ) {
        return -1;
      }
      
    } else if ( tmp == "<areafill>" ) {
      if ( this->readAreaFill( fin ) != 0 ) {
        return -1;
      }
      
    } else if ( tmp == "#" ) {
      fin.ignore(4098, '\n' );
      
    } else {
      return -1;
      
    }
  }
  return -1;
}



/** Updates this event
 */
void ModifyEvent::update()
{
  if ( this->status() == DynamicEvent::ACTIVE ) {
    DynamicEvent::m_counter -= 1;
    if ( DynamicEvent::m_counter < 0 ) {
      this->doModifications();
      this->status( DynamicEvent::COMPLETE );
      if ( DynamicEvent::m_sound != -1 ) {
        Sound::playSample( DynamicEvent::m_sound, false );
      }
    }
  }
}



/** Activates this event
 */
void ModifyEvent::activate()
{
  if (  this->isOkToActivate() ) {
    this->status( DynamicEvent::ACTIVE );
  } else {
    this->status( DynamicEvent::COMPLETE );
  }
}



/** Creates a modification.
 */
void ModifyEvent::createModification( istream& fin )
{
  string tmp;
  ModifyItem* pMI = new ModifyItem;
  pMI->blockX = 0;
  pMI->blockY = 0;
  pMI->layer = 0;
  pMI->newBlock = 0;
  
  fin >> pMI->blockX >> pMI->blockY;
  fin >> tmp;
  fin >> pMI->newBlock;
  
  m_modifications.push_back( pMI );
}



/** Creates a horizontal sequence of modifications.
 */
void ModifyEvent::createHLineModification( istream& fin )
{
  string tmp;
  int fromX;
  int fromY;
  int toX;
  int newBlock;
  
  fin >> fromX >> fromY >> toX;
  fin >> tmp;
  fin >> newBlock;
  
  for (int x=fromX; x < toX; x++) {
    ModifyItem* pMI = new ModifyItem;
    pMI->blockX = x;
    pMI->blockY = fromY;
    pMI->layer = 0;
    pMI->newBlock = newBlock;
    m_modifications.push_back( pMI );
  }
}



/** Creates a vertical sequence of modifications.
 */
void ModifyEvent::createVLineModification( istream& fin )
{
  string tmp;
  int fromX;
  int fromY;
  int toY;
  int newBlock;
  
  fin >> fromX >> fromY >> toY;
  fin >> tmp;
  fin >> newBlock;
  
  for (int y=fromY; y<toY; y++) {
    ModifyItem* pMI = new ModifyItem;
    pMI->blockX = fromX;
    pMI->blockY = y;
    pMI->layer = 0;
    pMI->newBlock = newBlock;
    m_modifications.push_back( pMI );
  }
}



/** Reads the copyblock element
 */
int ModifyEvent::readCopyBlock( istream& rIn )
{
  int fromX = 0;
  int fromY = 0;
  int fromLayer = 0;
  int toX = 0;
  int toY = 0;
  int toLayer = 0;
  int width = 0;
  int height = 0;
  
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    string tmp;
    rIn >> tmp;
    if ( tmp == "</copyblock>" ) {
      break;
      
    } else if ( tmp == "from:" ) {
      rIn >> fromX;
      rIn >> fromY;
      rIn >> fromLayer;
      
    } else if ( tmp == "to:" ) {
      rIn >> toX;
      rIn >> toY;
      
    } else if ( tmp == "to_layer:" ) {
      rIn >> toX >> toY >> toLayer;
      
    } else if ( tmp == "size:" ) {
      rIn >> width;
      rIn >> height;
      
    } else if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    }
  }

  // Do the modification elements
  Map::TLayerId lid = Map::int2LayerId( fromLayer );

  if ( Map::selectLayer( lid ) < 0 ) {
    return 0;
  }
  for ( int x=0; x < width; x++ ) {
    for ( int y=0; y < height; y++ ) {
      int blockNum = Map::blockNumberAt( fromX + x, fromY + y, Map::IN_BLOCKS );
      
      ModifyItem* pMI = new ModifyItem;
      pMI->blockX = toX + x;
      pMI->blockY = toY + y;
      pMI->layer = toLayer;
      pMI->newBlock = blockNum;
      m_modifications.push_back( pMI );
    }
  }
  Map::selectLayer( Map::EBackground );
  return 0;
}



/** Reads the areafill - element
 */
int ModifyEvent::readAreaFill( istream& rIn )
{
  int fromX = 0;
  int fromY = 0;
  int width = 0;
  int height = 0;
  int layernum = 0;
  int fillTile = 0;
  
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    string tmp;
    rIn >> tmp;
    if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "</areafill>" ) {
      break;
      
    } else if ( tmp == "from:" ) {
      rIn >> fromX >> fromY;
      
    } else if ( tmp == "size:" ) {
      rIn >> width >> height;
      
    } else if ( tmp == "layer:" ) {
      rIn >> layernum;
      
    } else if ( tmp == "tile:" ) {
      rIn >> fillTile;
      
    }
  }
  
  // Do the modify items
  for ( int w=0; w < width; w++ ) {
    for ( int h=0; h < height; h++ ) {
      ModifyItem* pMI = new ModifyItem;
      pMI->blockX = fromX + w;
      pMI->blockY = fromY + h;
      pMI->layer = layernum;
      pMI->newBlock = fillTile;
      m_modifications.push_back( pMI );
    }
  }
  return 0;
}



/** Does the modifications
 */
void ModifyEvent::doModifications()
{
  for (int i=0; i < m_modifications.size(); i++) {
    ModifyItem* pMI = m_modifications.at(i);
    if ( pMI != 0 ) {
      Map::TLayerId lid = Map::int2LayerId( pMI->layer );
      Map::selectLayer( lid );
      Map::setBlock( pMI->blockX, pMI->blockY, Map::IN_BLOCKS, pMI->newBlock );
    }
  }
  Map::selectLayer( Map::EBackground );
}


} // end of namespace

